let currentRecordGlobal = null;
let currentDailyDataGlobal = null;
let comparisonMode = 'litres'; // 'litres' or 'revenue'

document.addEventListener('DOMContentLoaded', () => {
    const urlParams = new URLSearchParams(window.location.search);
    const recordId = parseInt(urlParams.get('id'));

    if(!recordId) {
        alert("No record selected");
        window.location.href = 'sponsor.html';
        return;
    }

    // MILK_RECORDS is available because app.js is included
    const record = MILK_RECORDS.find(r => r.id === recordId);

    if (!record) {
        document.querySelector('.container').innerHTML = '<h2>Record Not Found</h2>';
        return;
    }

    // Store globally for toggle usage
    currentRecordGlobal = record;
    currentDailyDataGlobal = generateDailyData(record);

    renderHeader(record);
    renderCircularStats(currentDailyDataGlobal);
    renderLiquidGraph(currentDailyDataGlobal);
    renderComparisonBars(record, currentDailyDataGlobal); // Initial render (litres)
    renderDailyTable(record, currentDailyDataGlobal);
    renderMobileCards(record, currentDailyDataGlobal);
});

// ---------------------------------------
// TOGGLE FUNCTION
// ---------------------------------------
function toggleComparison(mode) {
    comparisonMode = mode;
    
    // Update Button Styles
    document.getElementById('btnLitres').classList.toggle('active', mode === 'litres');
    document.getElementById('btnRevenue').classList.toggle('active', mode === 'revenue');
    
    // Re-render Chart
    if(currentRecordGlobal && currentDailyDataGlobal) {
        renderComparisonBars(currentRecordGlobal, currentDailyDataGlobal);
    }
}

// ---------------------------------------
// RENDER BUTTERFLY CHART (Dynamic Colors & Toggle)
// ---------------------------------------
function renderComparisonBars(currentRecord, currentDailyData) {
    const container = document.getElementById('comparisonBars');
    
    // Find previous week's record
    const currentIndex = MILK_RECORDS.findIndex(r => r.id === currentRecord.id);
    const previousRecord = currentIndex < MILK_RECORDS.length - 1 ? 
        MILK_RECORDS[currentIndex + 1] : null;
    
    if (!previousRecord) {
        container.innerHTML = '<p style="color:var(--text-muted); text-align:center;">No previous week data available</p>';
        return;
    }
    
    const previousDailyData = generateDailyData(previousRecord);
    
    // Determine Price per Litre (Assuming constant for comparison simplicity, or use record specific)
    const curPrice = currentRecord.pricePerLitre;
    const prevPrice = previousRecord.pricePerLitre;

    // Calculate Values based on Mode
    const getVal = (data, price) => comparisonMode === 'revenue' ? (data.litres * price) : data.litres;
    const getLabel = (val) => comparisonMode === 'revenue' ? `KSh ${val.toLocaleString()}` : `${val}L`;

    // Find max value for scaling
    const currentValues = currentDailyData.map(d => getVal(d, curPrice));
    const prevValues = previousDailyData.map(d => getVal(d, prevPrice));
    
    const allValues = [...currentValues, ...prevValues];
    const maxValue = Math.max(...allValues) * 1.1; // 10% buffer
    
    // Header (Colors match the Left/Right gradient logic implicitly)
    let html = `
        <div class="butterfly-container">
            <div class="butterfly-header">
                <div style="text-align:right; flex:1; color:var(--text-muted);">
                    <i class="fas fa-caret-left"></i> LAST WEEK
                </div>
                <div style="width:60px; display:flex; justify-content:center;">
                    <div class="vs-circle">VS</div>
                </div>
                <div style="text-align:left; flex:1; color:#fff;">
                    THIS WEEK <i class="fas fa-caret-right"></i>
                </div>
            </div>
    `;
    
    // Loop rows
    for (let i = 0; i < 7; i++) {
        const currentDay = currentDailyData[i];
        const previousDay = previousDailyData[i];
        
        const curVal = getVal(currentDay, curPrice);
        const prevVal = getVal(previousDay, prevPrice);
        
        const curPct = (curVal / maxValue) * 100;
        const prevPct = (prevVal / maxValue) * 100;
        
        // Cycle between 0, 1, 2 for gradients (Pink, Purple, Cyan)
        const gradientIndex = i % 3; 

        html += `
            <div class="butterfly-row">
                <div class="col-left">
                    <div class="bf-bar left bf-grad-${gradientIndex}" style="width: ${prevPct}%;">
                        <span class="bf-val">${getLabel(prevVal)}</span>
                    </div>
                </div>

                <div class="butterfly-date">${currentDay.dayName}</div>

                <div class="col-right">
                    <div class="bf-bar right bf-grad-${gradientIndex}" style="width: ${curPct}%;">
                        <span class="bf-val">${getLabel(curVal)}</span>
                    </div>
                </div>
            </div>
        `;
    }
    
    html += '</div>';
    container.innerHTML = html;
}

// ---------------------------------------
// GENERATE DAILY DATA
// ---------------------------------------
function generateDailyData(r) {
    const days = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    const datePart = r.dateRange.split('/')[0]; 
    const startDayInt = parseInt(datePart.split('-')[0]); 

    let remainingLitres = r.litres;
    let remainingAdvance = r.advanceTaken;
    let data = [];

    for(let i=0; i<7; i++) {
        // Litres Logic
        let dayLitres = 0;
        if(i === 6) {
            dayLitres = remainingLitres;
        } else {
            let avg = Math.floor(r.litres / 7);
            let variance = Math.floor(Math.random() * 5) - 2;
            dayLitres = Math.max(0, avg + variance);
            if(remainingLitres - dayLitres < 0) dayLitres = remainingLitres;
            remainingLitres -= dayLitres;
        }

        // Advance Logic
        let dayAdvance = 0;
        if (remainingAdvance > 0 && (i === 1 || i === 4 || i === 6)) {
            if(i === 6) {
                dayAdvance = remainingAdvance;
            } else {
                let chunk = remainingAdvance >= 1000 ? 500 : remainingAdvance;
                if(Math.random() > 0.3) {
                    dayAdvance = chunk;
                    remainingAdvance -= chunk;
                }
            }
        }

        data.push({
            dayName: days[i],
            date: (startDayInt + i),
            litres: dayLitres,
            advance: dayAdvance
        });
    }
    return data;
}

// ---------------------------------------
// CIRCULAR STATS
// ---------------------------------------
function renderCircularStats(dailyData) {
    const container = document.getElementById('circularStats');
    const topDays = [...dailyData].sort((a, b) => b.litres - a.litres).slice(0, 3);
    const maxLitres = topDays[0].litres;
    const colors = ['#ff0080', '#7928ca', '#00cec9'];
    
    let html = '';
    topDays.forEach((day, index) => {
        const percentage = Math.round((day.litres / maxLitres) * 100);
        html += `
            <div class="circular-stat">
                <div class="circular-progress" style="--progress-value: ${percentage}%; --progress-color: ${colors[index]};">
                    <span class="circular-value">${day.litres}L</span>
                </div>
                <div class="circular-label">${day.dayName}</div>
            </div>
        `;
    });
    container.innerHTML = html;
}

// ---------------------------------------
// LIQUID GRAPH
// ---------------------------------------
function renderLiquidGraph(dailyData) {
    const container = document.getElementById('liquidGraph');
    let html = '';
    const maxVal = Math.max(...dailyData.map(d => d.litres));

    dailyData.forEach((d, index) => {
        let heightPct = (d.litres / maxVal) * 85;
        if (heightPct < 10) heightPct = 10;
        const colors = ["liq-purple", "liq-cyan", "liq-green", "liq-orange"];
        const colorClass = colors[index % colors.length];

        html += `
        <div class="tank-wrapper">
            <div class="tank-glass">
                <div class="tank-val">${d.litres}L</div>
                <div class="liquid ${colorClass}" style="height:${heightPct}%;"></div>
            </div>
            <div class="tank-label">${d.dayName}</div>
        </div>
        `;
    });
    container.innerHTML = html;
}

// ---------------------------------------
// HEADER
// ---------------------------------------
function renderHeader(r) {
    const startDay = parseInt(r.dateRange.split('-')[0]);
    let displayTitle = r.title;

    if(r.title !== "This Week" && r.title !== "Last Week") {
        if (startDay >= 1 && startDay <= 7)  displayTitle = '1st Week - ' + r.month;
        else if (startDay >= 8 && startDay <= 14) displayTitle = '2nd Week - ' + r.month;
        else if (startDay >= 15 && startDay <= 21) displayTitle = '3rd Week - ' + r.month;
        else if (startDay >= 22 && startDay <= 31) displayTitle = '4th Week - ' + r.month;
    }

    document.getElementById('wkTitle').innerText = displayTitle;
    document.getElementById('wkRange').innerText = `${r.dateRange} • ${r.month} 2025`;
    
    let badgeClass = 'status-unpaid';
    if(r.status === 'Paid') badgeClass = 'status-paid';
    else if(r.status === 'Partially Paid') badgeClass = 'status-partial';
    
    document.getElementById('wkStatusBadge').innerHTML =
        `<span class="status-badge ${badgeClass}" style="font-size:0.9rem;">${r.status}</span>`;

    const revenue = r.litres * r.pricePerLitre;
    const net = revenue - r.advanceTaken;

    document.getElementById('wkLitres').innerText = r.litres + " L";
    document.getElementById('wkRevenue').innerText = "KSh " + revenue.toLocaleString();
    document.getElementById('wkAdvances').innerText = "KSh " + r.advanceTaken.toLocaleString();
    document.getElementById('wkNet').innerText = "KSh " + net.toLocaleString();
}

// ---------------------------------------
// DAILY TABLE & CARDS
// ---------------------------------------
function renderDailyTable(r, dailyData) {
    const tbody = document.getElementById('dailyTableBody');
    let html = '';

    dailyData.forEach(d => {
        const gross = d.litres * r.pricePerLitre;
        const net = gross - d.advance;
        const netStyle = net < 0 ? 'color:#ff7675' : 'color:#00b894';

        html += `
        <tr>
            <td>
                <div style="font-weight:bold; color:#fff;">${d.dayName}</div>
                <div style="font-size:0.75rem; color:var(--text-muted);">${d.date} ${r.month}</div>
            </td>
            <td><span style="color:#00cec9; font-weight:bold;">${d.litres} L</span></td>
            <td>${r.pricePerLitre}</td>
            <td style="text-align:right;">${gross.toLocaleString()}</td>
            <td style="text-align:right; color:#fdcb6e;">${d.advance > 0 ? d.advance.toLocaleString() : '-'}</td>
            <td style="text-align:right; font-weight:bold; ${netStyle}">${net.toLocaleString()}</td>
        </tr>`;
    });

    const totalRev = r.litres * r.pricePerLitre;
    const totalNet = totalRev - r.advanceTaken;

    html += `
        <tr style="background:rgba(255,255,255,0.05); border-top:2px solid #444;">
            <td style="color:#fff; font-weight:700;">WEEKLY TOTALS</td>
            <td style="color:#00cec9; font-weight:700;">${r.litres} L</td>
            <td>-</td>
            <td style="text-align:right; font-weight:700;">${totalRev.toLocaleString()}</td>
            <td style="text-align:right; color:#fdcb6e; font-weight:700;">${r.advanceTaken.toLocaleString()}</td>
            <td style="text-align:right; color:#00b894; font-weight:700;">${totalNet.toLocaleString()}</td>
        </tr>`;

    tbody.innerHTML = html;
}

function renderMobileCards(r, dailyData) {
    const container = document.getElementById('dailyCardsView');
    let html = '';

    dailyData.forEach(d => {
        const gross = d.litres * r.pricePerLitre;
        const net = gross - d.advance;
        const netColor = net < 0 ? '#ff7675' : '#00b894';

        html += `
            <div class="daily-card">
                <div class="card-header">
                    <div>
                        <div class="card-day">${d.dayName}</div>
                        <div class="card-date">${d.date} ${r.month}</div>
                    </div>
                    <div class="card-litres">${d.litres} L</div>
                </div>
                <div class="card-body">
                    <div class="card-field"><div class="card-field-label">Rate</div><div class="card-field-value">KSh ${r.pricePerLitre}</div></div>
                    <div class="card-field"><div class="card-field-label">Gross</div><div class="card-field-value">KSh ${gross.toLocaleString()}</div></div>
                    <div class="card-field"><div class="card-field-label">Advance</div><div class="card-field-value" style="color:#fdcb6e">${d.advance > 0 ? 'KSh ' + d.advance.toLocaleString() : '-'}</div></div>
                    <div class="card-field"><div class="card-field-label">Net Pay</div><div class="card-field-value" style="color:${netColor}">KSh ${net.toLocaleString()}</div></div>
                </div>
            </div>`;
    });

    const totalRev = r.litres * r.pricePerLitre;
    const totalNet = totalRev - r.advanceTaken;

    html += `
        <div class="daily-card" style="background:rgba(0,206,201,0.1); border-color:#00cec9;">
            <div class="card-header">
                <div><div class="card-day">WEEKLY TOTALS</div></div>
                <div class="card-litres">${r.litres} L</div>
            </div>
            <div class="card-body">
                <div class="card-field"><div class="card-field-label">Total Revenue</div><div class="card-field-value">KSh ${totalRev.toLocaleString()}</div></div>
                <div class="card-field"><div class="card-field-label">Total Advances</div><div class="card-field-value" style="color:#fdcb6e">KSh ${r.advanceTaken.toLocaleString()}</div></div>
                <div class="card-field" style="grid-column: 1 / -1;"><div class="card-field-label">Net Pay</div><div class="card-field-value" style="color:#00b894; font-size:1.5rem;">KSh ${totalNet.toLocaleString()}</div></div>
            </div>
        </div>`;

    container.innerHTML = html;
}
