/* --- MILK DATA MODEL --- */
const MILK_RECORDS = [
    { id: 101, title: "This Week", month: "December", dateRange: "01-07/12/2025", status: "Not Paid", litres: 42, pricePerLitre: 45, advanceTaken: 1500, collector: "Florida" },
    { id: 102, title: "Last Week", month: "November", dateRange: "24-30/11/2025", status: "Partially Paid", litres: 43, pricePerLitre: 45, advanceTaken: 2000, collector: "Florida" },
    { id: 103, title: "Week 3", month: "November", dateRange: "17-23/11/2025", status: "Paid", litres: 41, pricePerLitre: 45, advanceTaken: 1000, collector: "Florida" },
    { id: 104, title: "Week 2", month: "November", dateRange: "10-16/11/2025", status: "Paid", litres: 44, pricePerLitre: 45, advanceTaken: 800, collector: "Florida" },
    { id: 105, title: "Week 1", month: "November", dateRange: "03-09/11/2025", status: "Paid", litres: 43, pricePerLitre: 45, advanceTaken: 1200, collector: "Florida" }
];

document.addEventListener('DOMContentLoaded', () => {
    processRecordTitles();
    const urlParams = new URLSearchParams(window.location.search);
    const selectedMonth = urlParams.get('month');

    if (selectedMonth) {
        initMonthView(selectedMonth);
    } else {
        if(document.getElementById('recordsContainer')) { 
            initDairyDashboard();
        }
    }
});

function processRecordTitles() {
    for(let i=0; i<MILK_RECORDS.length; i++) {
        const record = MILK_RECORDS[i];
        if (record.title === "This Week" || record.title === "Last Week") continue;
        const startDay = parseInt(record.dateRange.split('-')[0]);
        let weekPosition = record.title; 
        if (startDay >= 1 && startDay <= 7) weekPosition = '1st Week';
        else if (startDay >= 8 && startDay <= 14) weekPosition = '2nd Week';
        else if (startDay >= 15 && startDay <= 21) weekPosition = '3rd Week';
        else if (startDay >= 22 && startDay <= 31) weekPosition = '4th Week';
        record.title = weekPosition; 
    }
}

// --- DASHBOARD & MONTH VIEWS ---
function initDairyDashboard() {
    document.getElementById('dashboardView').style.display = 'block';
    document.getElementById('monthDetailsView').style.display = 'none';

    const visibleRecords = MILK_RECORDS.filter(r => {
        const isRecent = r.title === "This Week" || r.title === "Last Week";
        const isUnpaid = r.status !== "Paid";
        return isRecent || isUnpaid;
    });
    renderRecords(visibleRecords, 'recordsContainer');
    calculateStats(MILK_RECORDS); 
    renderMonthlySummary();
}

function initMonthView(monthName) {
    document.getElementById('dashboardView').style.display = 'none';
    document.getElementById('monthDetailsView').style.display = 'block';
    document.getElementById('monthTitle').innerText = `${monthName} 2025 Records`;
    const monthRecords = MILK_RECORDS.filter(r => r.month === monthName);
    renderRecords(monthRecords, 'monthWeeksContainer');
}

/* --- ADVANCE REQUEST LOGIC --- */
function openAdvanceModal() {
    document.getElementById('advanceModal').classList.add('active');
    // Reset to Step 1
    document.getElementById('advanceStep1').style.display = 'block';
    document.getElementById('advanceStep2').style.display = 'none';
    document.getElementById('advanceAmount').value = '';
    
    // Lock phone number again in case it was unlocked previously
    const phoneInput = document.getElementById('mpesaNumber');
    phoneInput.setAttribute('readonly', true);
    phoneInput.style.borderBottom = "none";
}

function closeAdvanceModal() {
    document.getElementById('advanceModal').classList.remove('active');
}

function calculateFee() {
    const input = document.getElementById('advanceAmount').value;
    const amount = parseFloat(input);

    if(!amount || amount <= 0) {
        alert("Please enter a valid amount.");
        return;
    }
    if(amount > 5000) {
        alert("Maximum advance limit is KSh 5,000");
        return;
    }

    // 7% Calculation
    const fee = amount * 0.07;
    const net = amount - fee;

    // Update UI
    document.getElementById('confAmount').innerText = "KSh " + amount.toLocaleString();
    document.getElementById('confFee').innerText = "- KSh " + fee.toLocaleString();
    document.getElementById('confNet').innerText = "KSh " + net.toLocaleString();

    // Switch to Step 2
    document.getElementById('advanceStep1').style.display = 'none';
    document.getElementById('advanceStep2').style.display = 'block';
}

// Enable Editing of the Phone Number
function enablePhoneEdit() {
    const input = document.getElementById('mpesaNumber');
    input.removeAttribute('readonly');
    input.focus();
    input.style.borderBottom = "1px solid #00cec9"; // Visual cue
}

function backToStep1() {
    document.getElementById('advanceStep1').style.display = 'block';
    document.getElementById('advanceStep2').style.display = 'none';
}

function processFinalAdvance() {
    const phone = document.getElementById('mpesaNumber').value;
    if(phone.length < 10) {
        alert("Please check the M-Pesa number.");
        return;
    }

    closeAdvanceModal();
    
    // Show Loading
    document.getElementById('loadingOverlay').style.display = 'flex';
    
    // Simulate API delay
    setTimeout(() => {
        document.getElementById('loadingOverlay').style.display = 'none';
        
        // Show Success Toast
        const toast = document.getElementById('toast');
        toast.classList.add('show');
        setTimeout(() => toast.classList.remove('show'), 3000);

    }, 2500);
}


/* --- RENDERERS --- */
function renderRecords(records, containerId) {
    const container = document.getElementById(containerId);
    let html = '';
    if(records.length === 0) { container.innerHTML = '<div style="text-align:center; padding:20px; color:#666;">No records found.</div>'; return; }

    records.forEach(r => {
        const totalRevenue = r.litres * r.pricePerLitre;
        const netPayable = totalRevenue - r.advanceTaken;
        let badgeClass = 'status-unpaid';
        if(r.status === 'Paid') badgeClass = 'status-paid';
        else if(r.status === 'Partially Paid') badgeClass = 'status-partial';
        const paymentLabel = r.status === 'Paid' ? 'Net Paid' : 'Net Payable';
        let netColor = r.status === 'Paid' ? '#00b894' : '#00cec9'; 
        if(netPayable < 0) netColor = '#ff7675';

        html += `
        <div class="ad-card">
            <div style="text-align:center; font-size:0.75rem; color:#666; font-weight:bold; letter-spacing:1px; margin-bottom:5px; text-transform:uppercase;">${r.month}</div>
            <div class="ad-header">
                <div>
                    <h3 class="ad-title">${r.title}</h3>
                    <div style="font-size:0.75rem; color:#aaa; margin-top:4px; display:flex; align-items:center; gap:8px;">
                        <span><i class="far fa-calendar-alt"></i> ${r.dateRange}</span>
                        <span style="width:1px; height:10px; background:#444;"></span>
                        <span style="color:#48dbfb; font-weight:500;"><i class="fas fa-truck-pickup" style="font-size:0.7rem;"></i> ${r.collector}</span>
                    </div>
                </div>
                <span class="status-badge ${badgeClass}">${r.status}</span>
            </div>
            <div class="ad-stats-row">
                <div class="ad-stat-item"><span class="ad-stat-val" style="color:#00cec9">${r.litres} L</span><span class="ad-stat-lbl">Litres</span></div>
                <div class="ad-stat-item"><span class="ad-stat-val">KSh ${r.pricePerLitre}</span><span class="ad-stat-lbl">Rate</span></div>
                <div class="ad-stat-item"><span class="ad-stat-val" style="color:#fdcb6e">KSh ${formatShort(totalRevenue)}</span><span class="ad-stat-lbl">Revenue</span></div>
            </div>
            <div style="margin-top:0.5rem; padding:12px; background:rgba(0,0,0,0.25); border-radius:8px; font-size:0.85rem;">
                <div style="display:flex; justify-content:space-between; margin-bottom:6px;"><span style="color:var(--text-muted);">Total Revenue:</span><span style="color:#fff;">KSh ${formatMoney(totalRevenue)}</span></div>
                <div style="display:flex; justify-content:space-between; margin-bottom:6px;"><span style="color:var(--text-muted);">Total Advances:</span><span style="color:#fdcb6e;">KSh ${formatMoney(r.advanceTaken)}</span></div>
                <div style="border-top:1px solid rgba(255,255,255,0.1); margin-top:6px; padding-top:6px; display:flex; justify-content:space-between;"><span style="color:var(--text-muted);">${paymentLabel}:</span><span style="color:${netColor}; font-weight:700;">KSh ${formatMoney(netPayable)}</span></div>
            </div>
            <a href="analytics.html?id=${r.id}" class="btn-analytics-pop">View Breakdown</a>
        </div>`;
    });
    container.innerHTML = html;
}

function renderMonthlySummary() {
    const container = document.getElementById('monthlyContainer');
    const monthlyData = {};
    MILK_RECORDS.forEach(r => {
        if(!monthlyData[r.month]) { monthlyData[r.month] = { records: [], totalLitres: 0, totalRevenue: 0, totalAdvances: 0 }; }
        monthlyData[r.month].records.push(r);
        monthlyData[r.month].totalLitres += r.litres;
        monthlyData[r.month].totalRevenue += (r.litres * r.pricePerLitre);
        monthlyData[r.month].totalAdvances += r.advanceTaken;
    });
    
    let html = '';
    Object.keys(monthlyData).forEach(month => {
        const data = monthlyData[month];
        const netPayable = data.totalRevenue - data.totalAdvances;
        let allPaid = data.records.every(r => r.status === 'Paid');
        let status = allPaid ? 'Paid' : 'Pending';
        let badgeClass = allPaid ? 'status-paid' : 'status-unpaid';
        const collectorName = data.records[0].collector;

        html += `
        <div class="ad-card">
            <div style="text-align:center; font-size:0.75rem; color:#666; font-weight:bold; letter-spacing:1px; margin-bottom:5px; text-transform:uppercase;">Monthly Summary</div>
            <div class="ad-header">
                <div>
                    <h3 class="ad-title">${month} 2025</h3>
                    <div style="font-size:0.75rem; color:#aaa; margin-top:4px; display:flex; align-items:center; gap:8px;">
                        <span><i class="far fa-calendar-alt"></i> ${data.records.length} Weeks Recorded</span>
                        <span style="width:1px; height:10px; background:#444;"></span>
                        <span style="color:#48dbfb; font-weight:500;"><i class="fas fa-truck-pickup" style="font-size:0.7rem;"></i> ${collectorName}</span>
                    </div>
                </div>
                <span class="status-badge ${badgeClass}">${status}</span>
            </div>
            <div class="ad-stats-row">
                <div class="ad-stat-item"><span class="ad-stat-val" style="color:#00cec9">${data.totalLitres} L</span><span class="ad-stat-lbl">Total Litres</span></div>
                <div class="ad-stat-item"><span class="ad-stat-val">KSh 45</span><span class="ad-stat-lbl">Avg Price</span></div>
                <div class="ad-stat-item"><span class="ad-stat-val" style="color:#fdcb6e">KSh ${formatShort(data.totalRevenue)}</span><span class="ad-stat-lbl">Revenue</span></div>
            </div>
            <div style="margin-top:0.5rem; padding:12px; background:rgba(0,0,0,0.25); border-radius:8px; font-size:0.85rem;">
                <div style="display:flex; justify-content:space-between; margin-bottom:6px;"><span style="color:var(--text-muted);">Total Revenue:</span><span style="color:#fff;">KSh ${formatMoney(data.totalRevenue)}</span></div>
                <div style="display:flex; justify-content:space-between; margin-bottom:6px;"><span style="color:var(--text-muted);">Total Advances:</span><span style="color:#fdcb6e;">KSh ${formatMoney(data.totalAdvances)}</span></div>
                <div style="border-top:1px solid rgba(255,255,255,0.1); margin-top:6px; padding-top:6px; display:flex; justify-content:space-between;"><span style="color:var(--text-muted);">Net Payable:</span><span style="color:#00b894; font-weight:700;">KSh ${formatMoney(netPayable)}</span></div>
            </div>
            <a href="monthly_analytics.html?month=${month}" class="btn-analytics-pop">Full Month Report</a>
        </div>`;
    });
    container.innerHTML = html;
}

function calculateStats(records) {
    let pendingLitres = 0; let unpaidBalance = 0; let weekLitres = 0; let unrecoveredAdvances = 0;
    records.forEach(r => {
        const revenue = r.litres * r.pricePerLitre;
        const netPayable = revenue - r.advanceTaken;
        if(r.title === "This Week") weekLitres = r.litres;
        if(r.status !== "Paid") {
            pendingLitres += r.litres;
            unpaidBalance += netPayable;
            unrecoveredAdvances += r.advanceTaken;
        }
    });
    document.getElementById('statPendingCount').innerText = pendingLitres + " L";
    document.getElementById('statUnpaidBalance').innerText = "KSh " + formatMoney(unpaidBalance);
    document.getElementById('statWeekLitres').innerText = weekLitres.toLocaleString() + " L";
    document.getElementById('statAdvances').innerText = "KSh " + formatMoney(unrecoveredAdvances);
}

function formatMoney(num) { return num.toLocaleString('en-KE'); }
function formatShort(num) {
    if(num >= 1000000) { return (Math.floor(num / 100000) / 10) + 'M'; }
    if(num >= 1000) { return (Math.floor(num / 100) / 10) + 'k'; }
    return num.toLocaleString();
}
